<?php

namespace App\Http\Controllers;
use Illuminate\Http\Request;
use App\Models\Postulante;
use App\Models\Instituto;
use App\Models\Matricula;
use App\Models\ModuloFormativo;
use App\Models\PracticaProfesional;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\DB;


class CertificadoController extends Controller
{
    public function index()
    {
        return view('certificados.buscar'); // Coincide con la ruta del blade
    }

   public function buscar(Request $request)
{
    $dni = $request->input('dni');

    $postulante = Postulante::where('dni', $dni)->first();

    if (!$postulante) {
        return response()->json([]);
    }

    // Obtener la última matrícula del estudiante
    $ultimaMatricula = Matricula::where('postulante_id', $postulante->id_postulante)
        ->orderByDesc('fecha_matricula')
        ->first();

    // Si no tiene matrícula, no se puede emitir certificado
    if (!$ultimaMatricula) {
        return response()->json([
            'nombre' => "{$postulante->apellidos}, {$postulante->nombres}",
            'resultados' => [],
            'certificable' => false,
            'mensaje' => 'El estudiante no tiene matrícula registrada.'
        ]);
    }

    // Módulos del plan de estudios
    $modulosPlan = ModuloFormativo::where('id_plan_estudio', $ultimaMatricula->plan_estudio_id)
        ->pluck('id_modulo')
        ->toArray();

    // Módulos con práctica registrada
    $modulosRegistrados = PracticaProfesional::where('postulante_id', $postulante->id_postulante)
        ->pluck('id_modulo')
        ->toArray();

    // Verificar si faltan módulos
    $faltanModulos = array_diff($modulosPlan, $modulosRegistrados);
    $completo = count($faltanModulos) === 0;

    // Obtener las notas y cursos
    $resultados = DB::table('matriculas as m')
        ->join('postulantes as p', 'p.id_postulante', '=', 'm.postulante_id')
        ->join('matricula_cursos as mc', 'mc.matricula_id', '=', 'm.id')
        ->join('cursos as c', 'c.id', '=', 'mc.curso_id')
        ->join('semestres as s', 's.id', '=', 'm.semestre_id')
        ->join('periodos as pe', 'pe.id', '=', 'm.periodo_id')
        ->join('planes_estudio as pl', 'pl.id', '=', 'm.plan_estudio_id')
        ->leftJoin('nota_final as nf', function ($join) use ($postulante) {
            $join->on('nf.curso_id', '=', 'c.id')
                 ->where('nf.estudiante_id', '=', $postulante->id_postulante);
        })
        ->where('p.dni', $dni)
        ->orderBy('s.nombre')
        ->orderBy('c.nombre')
        ->get([
            'p.nombres',
            'p.apellidos',
            's.nombre as semestre',
            'c.nombre as curso',
            'c.creditos',
            'pe.nombre as periodo',
            'pl.nombre as plan_estudio',
            'nf.promedio as nota_final'
        ]);

    // Agregar campos adicionales a los resultados
    foreach ($resultados as $item) {
        $item->año = date('Y');
        $item->nota_final_letras = is_numeric($item->nota_final)
            ? $this->convertirNotaALetras((int)$item->nota_final)
            : '-';
    }

    return response()->json([
        'nombre' => "{$postulante->apellidos}, {$postulante->nombres}",
        'resultados' => $resultados,
        'certificable' => $completo,
        'mensaje' => $completo
            ? null
            : 'Faltan prácticas por registrar en todos los módulos del plan de estudios. No se puede emitir el certificado.'
    ]);
}


private function convertirNotaALetras($nota)
{
    $letras = [
        10 => "Diez",
        11 => "Once",
        12 => "Doce",
        13 => "Trece",
        14 => "Catorce",
        15 => "Quince",
        16 => "Dieciséis",
        17 => "Diecisiete",
        18 => "Dieciocho",
        19 => "Diecinueve",
        20 => "Veinte"
    ];

    return $letras[$nota] ?? "-";
}

public function pdf($dni)
{
    $postulante = Postulante::where('dni', $dni)->firstOrFail();

    $resultados = DB::table('matriculas as m')
        ->join('postulantes as p', 'p.id_postulante', '=', 'm.postulante_id')
        ->join('matricula_cursos as mc', 'mc.matricula_id', '=', 'm.id')
        ->join('cursos as c', 'c.id', '=', 'mc.curso_id')
        ->join('semestres as s', 's.id', '=', 'm.semestre_id')
        ->join('periodos as pe', 'pe.id', '=', 'm.periodo_id')
        ->join('planes_estudio as pl', 'pl.id', '=', 'm.plan_estudio_id')
        ->leftJoin('nota_final as nf', function ($join) use ($postulante) {
            $join->on('nf.curso_id', '=', 'c.id')
                 ->where('nf.estudiante_id', '=', $postulante->id_postulante);
        })
        ->where('p.dni', $dni)
        ->orderBy('s.nombre')
        ->orderBy('c.nombre')
        ->get([
            'p.nombres',
            'p.apellidos',
            'p.dni',
            's.nombre as semestre',
            'c.nombre as curso',
            'c.creditos',
            'pe.nombre as periodo',
            'pl.nombre as plan_estudio',
            'nf.promedio as nota_final'
        ]);

    foreach ($resultados as $item) {
        $item->año = date('Y');
        $item->nota_final_letras = is_numeric($item->nota_final)
            ? $this->convertirNotaALetras((int)$item->nota_final)
            : '-';
    }

    $instituto = DB::table('instituto')->first(); // Asumiendo que solo hay uno

    $pdf = Pdf::loadView('certificados.certificado_estudios', [
        'postulantes' => $resultados,
        'nombre_instituto' => $instituto->nombre ?? 'INSTITUTO'
        
    ]);

    return $pdf->stream('certificado_estudios.pdf');
}

}