<?php

namespace App\Http\Controllers;

use App\Models\Sesion;
use App\Models\Curso;
use Illuminate\Http\Request;
use App\Models\CursoDocente;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon; // Importar la clase Carbon para trabajar con fechas
class SesionController extends Controller
{
    public function index($curso_id)
    {
        $docente_id = Auth::user()->docente->id ?? null;
        $curso = Curso::findOrFail($curso_id);
        // Verifica si este curso le pertenece al docente autenticado
        $asignado = CursoDocente::where('curso_id', $curso_id)
                                ->where('docente_id', $docente_id)
                                ->exists();

        if (!$asignado) {
            abort(403, 'No tienes permiso para acceder a este curso.');
        }

        $sesiones = Sesion::where('curso_id', $curso_id)
                  ->orderBy('created_at', 'desc')
                  ->get();

        return view('docentes.sesiones.index', compact('sesiones', 'curso'));
    }

    public function store1(Request $request, $cursoId)
{
    // Validación
    $validated = $request->validate([
        'nombre' => 'required|string|max:255',
        'fecha' => 'required|date',
        'fecha_fin' => 'required|date',
        'archivo' => 'nullable|file|mimes:pdf,docx,jpg,png|max:10240', // Limitar a 10MB
        'estado' => 'required|boolean',
    ]);

    // Crear la nueva sesión asociada al curso
    $sesion = new Sesion([
        'nombre' => $validated['nombre'],
        'fecha' => $validated['fecha'],
        'fecha_fin' => $validated['fecha_fin'],
        'estado' => $validated['estado'],
        'curso_id' => $cursoId,
    ]);
    
    // Subir el archivo, si existe
    if ($request->hasFile('archivo')) {
        $archivo = $request->file('archivo');
        $nombreArchivo = time() . '_' . $archivo->getClientOriginalName(); // Generar nombre único
        $directorio = public_path('storage/sesiones');

        // Crear la carpeta si no existe
        if (!file_exists($directorio)) {
            mkdir($directorio, 0755, true);
        }

        // Mover el archivo a public/storage/sesiones
        $archivo->move($directorio, $nombreArchivo);

        // Asignar la ruta relativa al archivo
        $sesion->archivo = 'sesiones/' . $nombreArchivo;
    }

    // Guardar la sesión
    $sesion->save();

    return response()->json(['success' => true, 'message' => 'Sesión registrada correctamente']);
}
public function store(Request $request, $cursoId, \App\Services\MoodleService $moodle)
{
    // 1. VALIDACIÓN
    $validated = $request->validate([
        'nombre' => 'required|string|max:255',
        'fecha' => 'required|date',
        'fecha_fin' => 'required|date',
        'archivo' => 'nullable|file|mimes:pdf,docx,jpg,png|max:10240',
        'estado' => 'required|boolean',
    ]);

    // 2. OBTENER CURSO (Laravel + Moodle ID)
    $curso = Curso::findOrFail($cursoId);

    // 3. CREAR SESIÓN EN LARAVEL
    $sesion = new Sesion([
        'nombre' => $validated['nombre'],
        'fecha' => $validated['fecha'],
        'fecha_fin' => $validated['fecha_fin'],
        'estado' => $validated['estado'],
        'curso_id' => $cursoId,
    ]);

    // 4. SUBIR ARCHIVO LOCAL
    if ($request->hasFile('archivo')) {

        $archivo = $request->file('archivo');
        $nombreArchivo = time() . '_' . $archivo->getClientOriginalName();

        $directorio = public_path('storage/sesiones');
        if (!file_exists($directorio)) {
            mkdir($directorio, 0755, true);
        }

        $archivo->move($directorio, $nombreArchivo);

        $sesion->archivo = 'sesiones/' . $nombreArchivo;
    }

    // Guardamos la sesión en Laravel
    $sesion->save();

    // ========================================================
    //        🔥 5. CREAR LA SECCIÓN EN MOODLE AUTOMÁTICO 🔥
    // ========================================================

    if ($curso->moodle_course_id) {

        // Calcular número de sección: cantidad de sesiones + 1
        $sectionNum = Sesion::where('curso_id', $cursoId)->count();

        $respuesta = $moodle->crearSeccion(
            $curso->moodle_course_id,
            $sesion->nombre,        // Nombre de la sección
            $sectionNum             // Número de sección ordenado
        );

        // Guardamos el ID moodle si es necesario
        if (isset($respuesta[0]['id'])) {
            $sesion->moodle_section_id = $respuesta[0]['id'];
            $sesion->save();
        }
    }

    return response()->json([
        'success' => true,
        'message' => 'Sesión registrada correctamente',
        'sesion' => $sesion,
        'moodle' => $respuesta ?? null
    ]);
}


public function update(Request $request, $id)
{
    // Validar los datos del formulario
    $request->validate([
        'nombre' => 'required|string|max:255',
        'fecha' => 'required|date',
        'fecha_fin' => 'required|date',
        'archivo' => 'nullable|file|mimes:pdf,docx,jpg,png',  // Validar el archivo si es proporcionado
    ]);

    // Obtener la sesión a actualizar
    $sesion = Sesion::findOrFail($id);

    // Verificar si se subió un nuevo archivo
    if ($request->hasFile('archivo')) {
        // Eliminar el archivo anterior si existe
        if ($sesion->archivo && Storage::exists('public/' . $sesion->archivo)) {
            Storage::delete('public/' . $sesion->archivo);
        }

        // Subir el nuevo archivo
        $archivoPath = $request->file('archivo')->store('sesiones', 'public'); // Guardar en el directorio 'sesiones'
    } else {
        // Si no se subió un nuevo archivo, mantener el archivo anterior
        $archivoPath = $sesion->archivo;
    }

    // Actualizar la sesión con los nuevos datos
    $sesion->update([
        'nombre' => $request->nombre,
        'fecha' => $request->fecha,
        'fecha_fin' => $request->fecha_fin,
        'archivo' => $archivoPath, // Asignar el nuevo archivo (o el antiguo si no se subió uno nuevo)
    ]);

    // Retornar la respuesta (puedes enviar un mensaje de éxito o lo que necesites)
    return response()->json([
        'message' => 'La sesión fue actualizada correctamente.',
        'sesion' => $sesion,
    ]);
}

public function destroy($id)
{
    $sesion = Sesion::findOrFail($id);

    // Verificar si la sesión está activa (la fecha actual debe estar entre fecha y fecha_fin)
    $fechaActual = Carbon::now();
    if ($fechaActual->lt($sesion->fecha) || $fechaActual->gt($sesion->fecha_fin)) {
        return response()->json(['message' => 'No se puede eliminar la sesión. Debe estar activa y dentro del rango de fechas.'], 400);
    }

    // Eliminar archivo asociado si existe
    if ($sesion->archivo && Storage::exists('public/sesiones/' . $sesion->archivo)) {
        Storage::delete('public/sesiones/' . $sesion->archivo);
    }

    // Eliminar la sesión
    $sesion->delete();

    return response()->json(['message' => 'Sesión eliminada correctamente.']);
}


public function cambiarEstado(Request $request, $id)
{
    $request->validate([
        'activo' => 'required|boolean',
    ]);

    $sesion = Sesion::findOrFail($id);
    $sesion->activo = $request->activo;
    $sesion->save();

    return response()->json(['message' => 'Estado actualizado correctamente.']);
}


    // Aquí irán luego: store, edit, update, destroy
}
