<?php

namespace App\Http\Controllers;

use App\Models\ResultadoAdmision;
use App\Models\Postulante;
use App\Models\PlanEstudio;
use App\Models\Admision;
use Illuminate\Http\Request;

class ResultadoAdmisionController extends Controller
{
    /**
     * LISTAR RESULTADOS
     **/
    public function index(Request $request)
    {
        $resultados = ResultadoAdmision::with(['postulante', 'planEstudio', 'admision'])
            ->orderBy('id', 'DESC')
            ->paginate(15);

        return view('resultados.index', compact('resultados'));
    }

    /**
     * REGISTRAR NUEVO RESULTADO
     **/
   public function store(Request $request)
{
    $request->validate([
        'postulante_id'   => 'required|exists:postulantes,id_postulante',
        'plan_estudio_id' => 'required|exists:planes_estudio,id',
        'id_proceso'      => 'required|exists:admisiones,id_admision',
        'nota'            => 'required|numeric|min:0|max:20',
    ]);

    // VALIDAR DUPLICADO
    $existe = ResultadoAdmision::where('postulante_id', $request->postulante_id)
        ->where('id_proceso', $request->id_proceso)
        ->exists();

    if ($existe) {
        return response()->json([
            'errors' => [
                'postulante_id' => ['⚠ Este postulante ya tiene una nota registrada en este proceso de admisión.']
            ]
        ], 422);
    }

    // Determinar estado
    $estado = $request->nota >= 14 ? 'Vacante' : 'Sin Vacante';

    // Registrar
    $resultado = ResultadoAdmision::create([
        'postulante_id'   => $request->postulante_id,
        'plan_estudio_id' => $request->plan_estudio_id,
        'id_proceso'      => $request->id_proceso,
        'nota'            => $request->nota,
        'estado'          => $estado,
    ]);

    return response()->json([
        'success' => true,
        'message' => 'Nota registrada correctamente.',
        'data'    => $resultado
    ]);
}


    /**
     * MOSTRAR UN RESULTADO INDIVIDUAL
     **/
    public function show($id)
    {
        $resultado = ResultadoAdmision::with(['postulante', 'planEstudio', 'admision'])
            ->findOrFail($id);

        return response()->json($resultado);
    }

    /**
     * ACTUALIZAR RESULTADO
     **/
   public function update(Request $request, $id)
{
    $request->validate([
        'nota' => 'required|numeric|min:0|max:20',
    ]);

    $resultado = ResultadoAdmision::findOrFail($id);

    $resultado->nota = $request->nota;
    $resultado->estado = $request->nota >= 14 ? 'Vacante' : 'Sin Vacante';
    $resultado->save();

    return response()->json([
        'success' => true,
        'message' => 'Nota actualizada correctamente.',
        'data' => $resultado
    ]);
}


    /**
     * ELIMINAR RESULTADO
     **/
   public function destroy($id)
{
    try {
        $resultado = ResultadoAdmision::findOrFail($id);
        $resultado->delete();

        return response()->json([
            'success' => true,
            'message' => 'Resultado eliminado correctamente.'
        ]);

    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'No se pudo eliminar el resultado.',
            'error'   => $e->getMessage()
        ], 500);
    }
}


    /**
     * LISTAR RESULTADOS POR PROCESO DE ADMISIÓN
     **/
    public function porAdmision($admisionId)
    {
        $resultados = ResultadoAdmision::with(['postulante', 'planEstudio'])
            ->where('id_proceso', $admisionId)
            ->orderBy('nota', 'DESC')
            ->get();

        return response()->json($resultados);
    }

    /**
     * LISTAR POR PLAN DE ESTUDIO
     **/
    public function porPlan($planId)
    {
        $resultados = ResultadoAdmision::with(['postulante', 'admision'])
            ->where('plan_estudio_id', $planId)
            ->orderBy('nota', 'DESC')
            ->get();

        return response()->json($resultados);
    }

   public function importarExcel(Request $request)
{
    // VALIDAR ARCHIVO
    $request->validate([
        'archivo' => 'required|file|mimes:xlsx,xls|max:5120'
    ]);

    try {
        $import = new \App\Imports\ResultadosMasivoImport;

        // Importar archivo
        \Excel::import($import, $request->file('archivo'));

        // DEVOLVER SI TODO FUE OK
        return response()->json([
            'success' => true,
            'message' => 'Los resultados fueron importados correctamente.'
        ], 200, ['Content-Type' => 'application/json']);

    } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {

        // Errores fila por fila
        $failures = $e->failures();
        $errores = [];

        foreach ($failures as $failure) {
            $errores[] = "Fila {$failure->row()}: " . implode(", ", $failure->errors());
        }

        return response()->json([
            'success' => false,
            'errors' => $errores
        ], 422);

    } catch (\Exception $e) {

        return response()->json([
            'success' => false,
            'message' => 'Error inesperado: ' . $e->getMessage()
        ], 500);

    }
}


}
