<?php

namespace App\Imports;

use App\Models\Postulante;
use App\Models\Usuario;
use App\Models\UsuarioRol;
use App\Models\Role;
use App\Services\MoodleService;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;

class EstudiantesMasivoImport implements ToModel, WithHeadingRow, WithValidation
{
    protected $moodle;
    protected $dniUsadosExcel = []; // DNI duplicado dentro del archivo

    public function __construct()
    {
        $this->moodle = app(MoodleService::class);
    }

    // ============================================
    // VALIDACIONES DE COLUMNAS DEL EXCEL
    // ============================================
    public function rules(): array
    {
        return [
            '*.dni'       => 'required|digits:8',
            '*.nombres'   => 'required|string',
            '*.apellidos' => 'required|string',
            '*.email'     => 'nullable|email',
        ];
    }

    // ============================================
    // PROCESO PRINCIPAL
    // ============================================
    public function model(array $row)
    {
        // =======================
        // 1) Validar duplicado en el mismo Excel
        // =======================
        if (in_array($row['dni'], $this->dniUsadosExcel)) {
            throw new \Exception("❌ El DNI {$row['dni']} está duplicado dentro del archivo.");
        }

        $this->dniUsadosExcel[] = $row['dni'];

        // =======================
        // 2) Validar duplicado en BD
        // =======================
        if (Usuario::where('username', $row['dni'])->exists()) {
            throw new \Exception("❌ El usuario con DNI {$row['dni']} ya existe en el sistema.");
        }

        if (Postulante::where('dni', $row['dni'])->exists()) {
            throw new \Exception("❌ El estudiante con DNI {$row['dni']} ya está registrado.");
        }

        // =======================
        // 3) Obtener Rol Estudiante
        // =======================
        $rolEstudiante = Role::where('nombre', 'Estudiante')->first();
        if (!$rolEstudiante) {
            throw new \Exception("❌ El rol 'Estudiante' no existe.");
        }

        // =======================
        // 4) Crear usuario local
        // =======================
        $usuario = Usuario::create([
            'username'      => $row['dni'],
            'password_hash' => Hash::make($row['dni']),
            'status'        => 1
        ]);

        // Asignar rol estudiante
        UsuarioRol::create([
            'usuario_id' => $usuario->id,
            'rol_id'     => $rolEstudiante->id
        ]);

        // =======================
        // 5) Crear estudiante (Postulante)
        // =======================
        $postulante = Postulante::create([
            'codigo_postulante' => $row['dni'],  // 🔥 Nuevo campo
            'dni'               => $row['dni'],
            'nombres'           => $row['nombres'],
            'apellidos'         => $row['apellidos'],
            'email'             => $row['email'] ?? ($row['dni'] . '@noemail.com'),
            'telefono'          => $row['telefono'] ?? null,
            'grado'             => 'Estudiante',
            'usuario_id'        => $usuario->id
        ]);

        // =======================
        // 6) Crear usuario en Moodle
        // =======================
        $passwordMoodle = "Alumno@" . $row['dni'];

        $resp = $this->moodle->crearUsuario(
            $usuario->username,
            $passwordMoodle,
            $postulante->nombres,
            $postulante->apellidos,
            $postulante->email
        );

        if (isset($resp[0]['id'])) {
            $usuario->moodle_user_id = $resp[0]['id'];
            $usuario->save();
        }

        return $postulante;
    }
}
