<?php

namespace App\Http\Controllers;

use App\Models\Area;
use Illuminate\Http\Request;

class AreaController extends Controller
{
    public function index()
    {
        $areas = Area::orderBy('nombre')->paginate(10);
        return view('areas.index', compact('areas'));
    }

    public function create()
    {
        return view('areas.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => 'required|unique:areas,nombre|max:100',
            'descripcion' => 'nullable|string',
            'estado' => 'required|in:Activo,Inactivo',
        ]);

        try {
            $area = Area::create($validated);
            return response()->json([
                'success' => true,
                'message' => 'Área creada exitosamente',
                'area' => $area
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al crear el área',
                'error' => $e->getMessage()
            ]);
        }
    }


    public function edit(Area $area)
    {
        return view('areas.edit', compact('area'));
    }

    public function update(Request $request, Area $area)
{
    $validated = $request->validate([
        'nombre' => 'required|max:100|unique:areas,nombre,' . $area->id,
        'descripcion' => 'nullable|string',
        'estado' => 'required|in:Activo,Inactivo',
    ]);

    $area->update($validated);

    // Si la petición es AJAX o espera JSON, devuelve respuesta JSON
    if ($request->wantsJson()) {
        return response()->json([
            'success' => true,
            'message' => 'Área actualizada correctamente.'
        ]);
    }

    // En caso contrario, redirige con mensaje flash
    return redirect()->route('areas.index')->with('success', 'Área actualizada correctamente.');
}


    public function destroy(Area $area)
{
    try {
        $area->delete();

        return response()->json([
            'success' => true,
            'message' => 'Área eliminada correctamente.'
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error al eliminar el área.'
        ], 500);
    }
}

}
