<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\LogroCurso;
use App\Models\CursoDocente;
use App\Models\Docente;
use App\Models\Curso;
use Illuminate\Support\Facades\Auth;

class LogroCursoController extends Controller
{
    // Validar que el curso pertenezca al docente
    private function validarAcceso($curso_id)
    {
        $usuario = Auth::user();
        $docente = Docente::where('usuario_id', $usuario->id)->first();
        $cursoDocente = CursoDocente::where('curso_id', $curso_id)->where('docente_id', $docente->id)->first();

        if (!$cursoDocente) {
            abort(403, 'No tienes acceso a este curso.');
        }
    }

    public function index($curso_id)
    {
        $this->validarAcceso($curso_id);

        $logros = LogroCurso::where('curso_id', $curso_id)->get();
        $curso = Curso::findOrFail($curso_id);

        return view('docentes.logros.index', compact('logros', 'curso'));
    }

    public function create($curso_id)
    {
        $this->validarAcceso($curso_id);
        $curso = Curso::findOrFail($curso_id);

        return view('docentes.logros.create', compact('curso'));
    }

    public function store(Request $request, $curso_id)
    {
        $this->validarAcceso($curso_id);

        $request->validate([
            'nombre' => 'required|string|max:100',
            'descripcion' => 'nullable|string',
        ]);

        $logro = LogroCurso::create([
            'curso_id' => $curso_id,
            'nombre' => $request->nombre,
            'descripcion' => $request->descripcion,
        ]);

        // Retornamos JSON para la respuesta AJAX
        return response()->json([
            'message' => 'Logro creado correctamente.',
            'logro' => $logro,
        ]);
    }


    public function edit($id)
    {
        $logro = LogroCurso::findOrFail($id);
    
        // Validar acceso por relación con el curso
        $this->validarAcceso($logro->curso_id);
    
        return response()->json($logro); // Devuelve JSON para AJAX
    }

    public function update(Request $request, $id)
{
    $logro = LogroCurso::findOrFail($id);
    $this->validarAcceso($logro->curso_id);

    $request->validate([
        'nombre' => 'required|string|max:100',
        'descripcion' => 'nullable|string'
    ]);

    $logro->update([
        'nombre' => $request->nombre,
        'descripcion' => $request->descripcion,
    ]);

    return response()->json(['message' => 'Logro actualizado correctamente.']);
}

    
public function destroy($id)
{
    $logro = LogroCurso::findOrFail($id);
    $this->validarAcceso($logro->curso_id); // Aún validamos que el docente tenga acceso al curso

    $logro->delete();

    return response()->json([
        'success' => true,
        'message' => 'Logro eliminado correctamente.'
    ]);
}

}
