<?php

namespace App\Http\Controllers;

use App\Models\Tramite;
use Illuminate\Http\Request;
use App\Models\RequisitoTramite;
use App\Models\SolicitudTramite;
use App\Models\Area;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
class TramiteController extends Controller
{
    // Mostrar la lista de trámites
    public function index()
    {
        $tramites = Tramite::orderBy('created_at', 'desc')->get();
        $requisitos = RequisitoTramite::orderBy('descripcion')->get(); // puedes filtrar solo los activos si deseas
        return view('tramitess.index', compact('tramites', 'requisitos'));
    }

    // Guardar un nuevo trámite (puede usarse para AJAX)
    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:100|unique:tramites,nombre',
            'descripcion' => 'nullable|string|max:200',
            'estado' => 'required|in:Activo,Inactivo',
            'costo' => 'nullable|numeric|min:0',
            'tiempo' => 'nullable|string|max:50',
            'requisitos' => 'nullable|array',
            'requisitos.*' => 'exists:requisitos_tramite,id',
        ]);

        // Crear el trámite con todos los campos
        $tramite = Tramite::create([
            'nombre' => $request->nombre,
            'descripcion' => $request->descripcion,
            'estado' => $request->estado,
            'costo' => $request->costo,
            'tiempo' => $request->tiempo,
        ]);

        // Asociar requisitos si se envían
        if ($request->has('requisitos')) {
            $tramite->requisitos()->sync($request->input('requisitos'));
        }

        return response()->json([
            'message' => 'Trámite registrado correctamente.',
            'data' => $tramite
        ]);
    }



    // Obtener un trámite específico (para edición vía AJAX)
    public function show($id)
{
    $tramite = Tramite::with('requisitos')->findOrFail($id);

    return response()->json([
        'tramite'    => $tramite,
        'requisitos' => $tramite->requisitos->pluck('id')
    ]);
}



    // Actualizar trámite (usado en AJAX para modal de edición)
    public function update(Request $request, $id)
{
    $tramite = Tramite::findOrFail($id);

    $request->validate([
        'nombre' => 'required|string|max:255|unique:tramites,nombre,' . $tramite->id,
        'descripcion' => 'nullable|string',
        'estado' => 'required|in:Activo,Inactivo',
        'costo' => 'nullable|numeric',
        'tiempo' => 'nullable|string|max:50',
        'requisitos' => 'nullable|array',
        'requisitos.*' => 'exists:requisitos_tramite,id',
    ]);

    $tramite->update($request->only(['nombre', 'descripcion', 'estado', 'costo', 'tiempo']));

    // Sincronizar requisitos si vienen
    if ($request->has('requisitos')) {
        $tramite->requisitos()->sync($request->input('requisitos'));
    } else {
        // Si no se envían requisitos, se eliminan todos los actuales
        $tramite->requisitos()->detach();
    }

    return response()->json([
        'message' => 'Trámite actualizado correctamente.',
        'data' => $tramite
    ]);
}


    // Eliminar trámite
   public function destroy($id)
{
    $tramite = Tramite::findOrFail($id);
    $tramite->requisitos()->detach(); // Opcional, si usas tabla pivote
    $tramite->delete();

    return response()->json([
        'message' => 'Trámite eliminado correctamente.'
    ]);
}

public function derivar(Request $request)
{
    $request->validate([
        'solicitud_id' => 'required|exists:solicitudes_tramites,id',
        'area_destino_id' => 'required|exists:areas,id',
        'observacion' => 'nullable|string',
        'archivo_derivacion' => 'nullable|file|mimes:pdf,docx,png,jpg|max:2048',
    ]);

    $solicitud = DB::table('solicitudes_tramites')->where('id', $request->solicitud_id)->first();
    $usuario = auth()->user();

    if (!$solicitud) {
        return response()->json(['message' => 'Trámite no encontrado.'], 404);
    }

    // Guardar archivo si existe
    $archivoNombre = null;
    if ($request->hasFile('archivo_derivacion')) {
        $archivo = $request->file('archivo_derivacion');
        $archivoNombre = time() . '_' . $archivo->getClientOriginalName();
        $archivo->move(public_path('tramites/derivaciones'), $archivoNombre); // No usar storage/
    }

    // Registrar en historial
    DB::table('historial_derivaciones')->insert([
        'solicitud_id' => $solicitud->id,
        'area_origen_id' => $solicitud->area_id,
        'area_destino_id' => $request->area_destino_id,
        'estado_en_origen' => 'derivado',
        'estado_en_destino' => 'pendiente',
        'observacion' => $request->observacion,
        'usuario_id' => $usuario->id,
        'archivo_derivacion' => $archivoNombre,
        'created_at' => now(),
        'updated_at' => now(),
    ]);

    // Actualizar la solicitud
    DB::table('solicitudes_tramites')
        ->where('id', $solicitud->id)
        ->update([
            'estado' => 'derivado',
            'area_id' => $request->area_destino_id,
            'updated_at' => now(),
        ]);

    return response()->json(['message' => 'Trámite derivado correctamente.']);
}

public function rechazar(Request $request)
{
    $request->validate([
        'solicitud_id' => 'required|exists:solicitudes_tramites,id',
        'motivo' => 'required|string|max:1000',
    ]);

    $solicitud = SolicitudTramite::findOrFail($request->solicitud_id);
    $solicitud->estado = 'rechazado';
    $solicitud->motivo_rechazo = $request->motivo; // si tienes este campo
    $solicitud->save();

    return response()->json(['message' => 'Trámite rechazado correctamente.']);
}

public function finalizar(Request $request)
{
    $request->validate([
        'solicitud_id' => 'required|exists:solicitudes_tramites,id'
    ]);

    $solicitud = SolicitudTramite::find($request->solicitud_id);

    if (in_array($solicitud->estado, ['completado', 'rechazado'])) {
        return response()->json(['message' => 'Este trámite ya fue cerrado.'], 400);
    }

    $usuario = auth()->user();

    // Actualiza el estado de la solicitud
    $solicitud->estado = 'completado';
    $solicitud->fecha_resolucion = now();
    $solicitud->save();

    // Registra en historial
    HistorialDerivacion::create([
        'solicitud_id' => $solicitud->id,
        'area_origen_id' => DB::table('usuario_areas')
            ->where('usuario_id', $usuario->id)
            ->where('activo', 1)
            ->value('area_id'),
        'area_destino_id' => null,
        'estado_en_origen' => 'completado',
        'estado_en_destino' => 'completado',
        'observacion' => 'Trámite finalizado.',
        'usuario_id' => $usuario->id,
        'fecha_derivacion' => now()
    ]);

    return response()->json(['message' => 'Trámite finalizado correctamente.']);
}

public function indexPorArea()
{
    $areaActualId = auth()->user()->areas->first()?->id;


// Obtener los trámites asignados al área del usuario actual
$tramites = SolicitudTramite::with([
        'postulante:id_postulante,nombres,apellidos', // solo cargamos los campos necesarios
        'tramite:id,nombre'                          // solo cargamos los campos necesarios
    ])
    ->where('area_id', $areaActualId)
    ->whereIn('estado', ['pendiente', 'derivado'])
    ->orderByDesc('created_at')
    ->get();
$areas = Area::where('id', '!=', $areaActualId)->get(); // evitar derivarse a sí misma
// Retornar la vista con los datos
return view('areas.tramites.index', compact('tramites','areas'));

}

public function trazabilidad($id)
{
    $historial = DB::table('historial_derivaciones AS h')
        ->join('areas AS origen', 'h.area_origen_id', '=', 'origen.id')
        ->join('areas AS destino', 'h.area_destino_id', '=', 'destino.id')
        ->join('usuarios AS u', 'h.usuario_id', '=', 'u.id')
        ->leftJoin('docentes AS d', 'd.usuario_id', '=', 'u.id') // Para obtener nombre y apellido si es docente
        ->where('h.solicitud_id', $id)
        ->orderBy('h.fecha_derivacion', 'asc')
        ->get([
            'h.*',
            'origen.nombre AS area_origen',
            'destino.nombre AS area_destino',
            DB::raw("CONCAT(d.nombre, ' ', d.apellido) AS usuario_nombre")
        ]);

    $html = view('areas.tramites.partials.trazabilidad', compact('historial'))->render();

    return response()->json(['html' => $html]);
}


public function completados()
{
    $tramites = DB::table('solicitudes_tramites AS s')
        ->join('postulantes AS p', 's.postulante_id', '=', 'p.id_postulante')
        ->join('tramites AS t', 's.tramite_id', '=', 't.id')
        ->join('areas AS a', 's.area_id', '=', 'a.id')
        ->where('s.estado', 'completado')
        ->select(
            's.*',
            'p.nombres',
            'p.apellidos',
            't.nombre AS tipo_tramite',
            'a.nombre AS area'
        )
        ->orderByDesc('s.updated_at')
        ->get();

    return view('reportes.tramites.completados', compact('tramites'));
}


public function rechazados(Request $request)
{

    $tramites = DB::table('solicitudes_tramites AS s')
        ->join('postulantes AS p', 's.postulante_id', '=', 'p.id_postulante')
        ->join('tramites AS t', 's.tramite_id', '=', 't.id')
        ->join('areas AS a', 's.area_id', '=', 'a.id')
        ->where('s.estado', 'rechazado')
        ->select(
            's.*',
            'p.nombres',
            'p.apellidos',
            't.nombre AS tipo_tramite',
            'a.nombre AS area'
        )
        ->orderByDesc('s.updated_at')
        ->get();

    return view('reportes.tramites.rechazados', compact('tramites'));
    
}


}
