<?php

namespace App\Imports;

use App\Models\Docente;
use App\Models\Usuario;
use App\Models\UsuarioRol;
use App\Models\Role;
use App\Services\MoodleService;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;

class DocentesMasivoImport implements ToModel, WithHeadingRow, WithValidation
{
    protected $moodle;
    protected $dniUsadosEnExcel = [];

    public function __construct()
    {
        $this->moodle = app(MoodleService::class);
    }

    /**
     * VALIDACIÓN DE CADA FILA DEL EXCEL
     */
    public function rules(): array
    {
        return [
            '*.dni'         => 'required|digits:8',
            '*.nombre'      => 'required|string',
            '*.apellido'    => 'required|string',
            '*.email'       => 'nullable|email',
        ];
    }

    public function customValidationMessages()
    {
        return [
            '*.dni.required' => 'El DNI es obligatorio.',
            '*.dni.digits'   => 'El DNI debe tener exactamente 8 dígitos.',
            '*.nombre.required' => 'El nombre es obligatorio.',
        ];
    }

    public function model(array $row)
    {
        // =============================
        //  VALIDACIÓN 1: DNI repetido dentro del Excel
        // =============================
        if (in_array($row['dni'], $this->dniUsadosEnExcel)) {
            throw new \Exception("El DNI {$row['dni']} está duplicado dentro del archivo Excel.");
        }

        $this->dniUsadosEnExcel[] = $row['dni'];

        // =============================
        //  VALIDACIÓN 2: Usuario o docente ya existe
        // =============================
        if (Usuario::where('username', $row['dni'])->exists()) {
            throw new \Exception("El usuario con DNI {$row['dni']} ya existe en el sistema.");
        }

        if (Docente::where('dni', $row['dni'])->exists()) {
            throw new \Exception("El docente con DNI {$row['dni']} ya está registrado.");
        }

        // Obtener rol Docente
        $rolDocente = Role::where('nombre', 'Docente')->first();
        if (!$rolDocente) {
            throw new \Exception("El rol 'Docente' no existe. Debes crearlo primero.");
        }

        // =============================
        // 1) CREAR USUARIO LOCAL
        // =============================
        $usuario = Usuario::create([
            'username'      => $row['dni'],
            'password_hash' => Hash::make($row['dni']),
            'status'        => 'Disponible',
        ]);

        // =============================
        // 2) ASIGNAR ROL DOCENTE
        // =============================
        UsuarioRol::create([
            'usuario_id' => $usuario->id,
            'rol_id'     => $rolDocente->id
        ]);

        // =============================
        // 3) CREAR DOCENTE
        // =============================
        $docente = Docente::create([
            'usuario_id'   => $usuario->id,
            'nombre'       => $row['nombre'],
            'apellido'     => $row['apellido'],
            'dni'          => $row['dni'],
            'email'        => $row['email'] ?? ($row['dni'] . "@noemail.com"),
            'telefono'     => $row['telefono'] ?? null,
            'direccion'    => $row['direccion'] ?? null,
            'departamento' => $row['departamento'] ?? null,
            'cargo'        => $row['cargo'] ?? null,
        ]);

        // =============================
        // 4) CREAR USUARIO EN MOODLE
        // =============================

        $passwordMoodle = "Docente@" . $row['dni'];

        $resp = $this->moodle->crearUsuario(
            $usuario->username,
            $passwordMoodle,
            $docente->nombre,
            $docente->apellido,
            $docente->email
        );

        if (isset($resp[0]['id'])) {
            $usuario->moodle_user_id = $resp[0]['id'];
            $usuario->save();
        }

        return $docente;
    }
}
